#!/bin/sh
set -e
set -u
set -o pipefail

function on_error {
  echo "$(realpath -mq "${0}"):$1: error: Unexpected failure"
}
trap 'on_error $LINENO' ERR

if [ -z ${UNLOCALIZED_RESOURCES_FOLDER_PATH+x} ]; then
  # If UNLOCALIZED_RESOURCES_FOLDER_PATH is not set, then there's nowhere for us to copy
  # resources to, so exit 0 (signalling the script phase was successful).
  exit 0
fi

mkdir -p "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}"

RESOURCES_TO_COPY=${PODS_ROOT}/resources-to-copy-${TARGETNAME}.txt
> "$RESOURCES_TO_COPY"

XCASSET_FILES=()

# This protects against multiple targets copying the same framework dependency at the same time. The solution
# was originally proposed here: https://lists.samba.org/archive/rsync/2008-February/020158.html
RSYNC_PROTECT_TMP_FILES=(--filter "P .*.??????")

case "${TARGETED_DEVICE_FAMILY:-}" in
  1,2)
    TARGET_DEVICE_ARGS="--target-device ipad --target-device iphone"
    ;;
  1)
    TARGET_DEVICE_ARGS="--target-device iphone"
    ;;
  2)
    TARGET_DEVICE_ARGS="--target-device ipad"
    ;;
  3)
    TARGET_DEVICE_ARGS="--target-device tv"
    ;;
  4)
    TARGET_DEVICE_ARGS="--target-device watch"
    ;;
  *)
    TARGET_DEVICE_ARGS="--target-device mac"
    ;;
esac

install_resource()
{
  if [[ "$1" = /* ]] ; then
    RESOURCE_PATH="$1"
  else
    RESOURCE_PATH="${PODS_ROOT}/$1"
  fi
  if [[ ! -e "$RESOURCE_PATH" ]] ; then
    cat << EOM
error: Resource "$RESOURCE_PATH" not found. Run 'pod install' to update the copy resources script.
EOM
    exit 1
  fi
  case $RESOURCE_PATH in
    *.storyboard)
      echo "ibtool --reference-external-strings-file --errors --warnings --notices --minimum-deployment-target ${!DEPLOYMENT_TARGET_SETTING_NAME} --output-format human-readable-text --compile ${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename \"$RESOURCE_PATH\" .storyboard`.storyboardc $RESOURCE_PATH --sdk ${SDKROOT} ${TARGET_DEVICE_ARGS}" || true
      ibtool --reference-external-strings-file --errors --warnings --notices --minimum-deployment-target ${!DEPLOYMENT_TARGET_SETTING_NAME} --output-format human-readable-text --compile "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename \"$RESOURCE_PATH\" .storyboard`.storyboardc" "$RESOURCE_PATH" --sdk "${SDKROOT}" ${TARGET_DEVICE_ARGS}
      ;;
    *.xib)
      echo "ibtool --reference-external-strings-file --errors --warnings --notices --minimum-deployment-target ${!DEPLOYMENT_TARGET_SETTING_NAME} --output-format human-readable-text --compile ${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename \"$RESOURCE_PATH\" .xib`.nib $RESOURCE_PATH --sdk ${SDKROOT} ${TARGET_DEVICE_ARGS}" || true
      ibtool --reference-external-strings-file --errors --warnings --notices --minimum-deployment-target ${!DEPLOYMENT_TARGET_SETTING_NAME} --output-format human-readable-text --compile "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename \"$RESOURCE_PATH\" .xib`.nib" "$RESOURCE_PATH" --sdk "${SDKROOT}" ${TARGET_DEVICE_ARGS}
      ;;
    *.framework)
      echo "mkdir -p ${TARGET_BUILD_DIR}/${FRAMEWORKS_FOLDER_PATH}" || true
      mkdir -p "${TARGET_BUILD_DIR}/${FRAMEWORKS_FOLDER_PATH}"
      echo "rsync --delete -av "${RSYNC_PROTECT_TMP_FILES[@]}" $RESOURCE_PATH ${TARGET_BUILD_DIR}/${FRAMEWORKS_FOLDER_PATH}" || true
      rsync --delete -av "${RSYNC_PROTECT_TMP_FILES[@]}" "$RESOURCE_PATH" "${TARGET_BUILD_DIR}/${FRAMEWORKS_FOLDER_PATH}"
      ;;
    *.xcdatamodel)
      echo "xcrun momc \"$RESOURCE_PATH\" \"${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename "$RESOURCE_PATH"`.mom\"" || true
      xcrun momc "$RESOURCE_PATH" "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename "$RESOURCE_PATH" .xcdatamodel`.mom"
      ;;
    *.xcdatamodeld)
      echo "xcrun momc \"$RESOURCE_PATH\" \"${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename "$RESOURCE_PATH" .xcdatamodeld`.momd\"" || true
      xcrun momc "$RESOURCE_PATH" "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename "$RESOURCE_PATH" .xcdatamodeld`.momd"
      ;;
    *.xcmappingmodel)
      echo "xcrun mapc \"$RESOURCE_PATH\" \"${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename "$RESOURCE_PATH" .xcmappingmodel`.cdm\"" || true
      xcrun mapc "$RESOURCE_PATH" "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}/`basename "$RESOURCE_PATH" .xcmappingmodel`.cdm"
      ;;
    *.xcassets)
      ABSOLUTE_XCASSET_FILE="$RESOURCE_PATH"
      XCASSET_FILES+=("$ABSOLUTE_XCASSET_FILE")
      ;;
    *)
      echo "$RESOURCE_PATH" || true
      echo "$RESOURCE_PATH" >> "$RESOURCES_TO_COPY"
      ;;
  esac
}
if [[ "$CONFIGURATION" == "Debug" ]]; then
  install_resource "${PODS_ROOT}/APLongLinkService/APLongLinkService.framework/APLongLinkService.bundle"
  install_resource "${PODS_ROOT}/APMAftsCommon/APMAftsCommon.framework/APMAftsCommon.bundle"
  install_resource "${PODS_ROOT}/APMCDNImage/APMCDNImage.framework/APMCDNImage.bundle"
  install_resource "${PODS_ROOT}/APMFileTransfer/APMFileTransfer.framework/APMFileTransfer.bundle"
  install_resource "${PODS_ROOT}/APMHevc/APMHevc.framework/APMHevc.bundle"
  install_resource "${PODS_ROOT}/APMImageProcessing/APMImageProcessing.framework/APMImageProcessing.bundle"
  install_resource "${PODS_ROOT}/APMMediaCache/APMMediaCache.framework/APMMediaCache.bundle"
  install_resource "${PODS_ROOT}/APMUtils/APMUtils.framework/APMUtils.bundle"
  install_resource "${PODS_ROOT}/APMVideo/APMVideo.framework/APMVideo.bundle"
  install_resource "${PODS_ROOT}/APMVoiceManager/APMVoiceManager.framework/APMVoiceManager.bundle"
  install_resource "${PODS_ROOT}/APMap/APMap.framework/APMap.bundle"
  install_resource "${PODS_ROOT}/APMediaBasic/APMediaBasic.framework/APMediaBasic.bundle"
  install_resource "${PODS_ROOT}/APMobileFramework/APMobileFramework.framework/APMobileFramework.bundle"
  install_resource "${PODS_ROOT}/APMobileNetwork/APMobileNetwork.framework/APMobileNetwork.bundle"
  install_resource "${PODS_ROOT}/APMultimedia/APMultimedia.framework/APMultimedia.bundle"
  install_resource "${PODS_ROOT}/APRemoteLogging/APRemoteLogging.framework/APRemoteLogging.bundle"
  install_resource "${PODS_ROOT}/AUBase/AUBase.framework/AUBase.bundle"
  install_resource "${PODS_ROOT}/AUKeyboard/AUKeyboard.framework/AUKeyboard.bundle"
  install_resource "${PODS_ROOT}/AUTheme/AUTheme.framework/AUTheme.bundle"
  install_resource "${PODS_ROOT}/AliUpgradeCheckService/AliUpgradeCheckService.framework/AliUpgradeCheckService.bundle"
  install_resource "${PODS_ROOT}/AliUpgradeUI/AliUpgradeUI.framework/AliUpgradeUI.bundle"
  install_resource "${PODS_ROOT}/AntUI/AntUI.framework/APCommonUI.bundle"
  install_resource "${PODS_ROOT}/AntUIToken/AntUIToken.framework/AntUIToken.bundle"
  install_resource "${PODS_ROOT}/AriverComponent/AriverComponent.framework/AriverComponent.bundle"
  install_resource "${PODS_ROOT}/AriverKernel/AriverKernel.framework/AriverKernel.bundle"
  install_resource "${PODS_ROOT}/AriverMPNebulaAdapter/AriverMPNebulaAdapter.framework/MPInternalPlugins.bundle"
  install_resource "${PODS_ROOT}/AriverMPNebulaAdapter/AriverMPNebulaAdapter.framework/MPNebulaAdapter.bundle"
  install_resource "${PODS_ROOT}/AriverMap/AriverMap.framework/AriverMap.bundle"
  install_resource "${PODS_ROOT}/AriverNebulaAppBiz/AriverNebulaAppBiz.framework/NebulaAppBiz.bundle"
  install_resource "${PODS_ROOT}/AriverNebulaAppManager/AriverNebulaAppManager.framework/NebulaAppManager.bundle"
  install_resource "${PODS_ROOT}/AriverNebulaBiz/AriverNebulaBiz.framework/NebulaBiz.bundle"
  install_resource "${PODS_ROOT}/AriverNebulaBiz/AriverNebulaBiz.framework/NebulaPoseidon.bundle"
  install_resource "${PODS_ROOT}/AriverNebulaConfig/AriverNebulaConfig.framework/NebulaConfig.bundle"
  install_resource "${PODS_ROOT}/AriverRuntime/AriverRuntime.framework/AriverRuntime.bundle"
  install_resource "${PODS_ROOT}/AriverTinyAppCommon/AriverTinyAppCommon.framework/TinyAppCommon.bundle"
  install_resource "${PODS_ROOT}/AriverTinyappService/AriverTinyappService.framework/TinyappService.bundle"
  install_resource "${PODS_ROOT}/BEEAudioUtil/BEEAudioUtil.framework/BEEAudioUtil.bundle"
  install_resource "${PODS_ROOT}/BeeCapture/BeeCapture.framework/BeeCapture.bundle"
  install_resource "${PODS_ROOT}/BeeCityPicker/BeeCityPicker.framework/BeeCityPicker.bundle"
  install_resource "${PODS_ROOT}/BeeLocationPicker/BeeLocationPicker.framework/BeeLocationPicker.bundle"
  install_resource "${PODS_ROOT}/BeeMediaPlayer/BeeMediaPlayer.framework/BeeMediaPlayer.bundle"
  install_resource "${PODS_ROOT}/BeePhoto/BeePhoto.framework/BeePhoto.bundle"
  install_resource "${PODS_ROOT}/BeePhotoBrowser/BeePhotoBrowser.framework/BeePhotoBrowser.bundle"
  install_resource "${PODS_ROOT}/BeePhotoPicker/BeePhotoPicker.framework/BeePhotoPicker.bundle"
  install_resource "${PODS_ROOT}/BeeUtils/BeeUtils.framework/BeeUtils.bundle"
  install_resource "${PODS_ROOT}/MAMapKit/MAMapKit.framework/AMap.bundle"
  install_resource "${PODS_ROOT}/MPHotpatchSDK/MPHotpatchSDK.framework/DynamicDeployment.bundle"
  install_resource "${PODS_ROOT}/MPOcr/MPOcr.framework/MPOcr.bundle"
  install_resource "${PODS_ROOT}/MPOcrCarLicense/MPOcrCarLicense.framework/MPOcrCarLicense.bundle"
  install_resource "${PODS_ROOT}/MPOcrDriverLicense/MPOcrDriverLicense.framework/MPOcrDriverLicense.bundle"
  install_resource "${PODS_ROOT}/MPOcrDrivingLicense/MPOcrDrivingLicense.framework/MPOcrDrivingLicense.bundle"
  install_resource "${PODS_ROOT}/MPOcrGasMeter/MPOcrGasMeter.framework/MPOcrGasMeter.bundle"
  install_resource "${PODS_ROOT}/MPOcrPassPort/MPOcrPassPort.framework/MPOcrPassPort.bundle"
  install_resource "${PODS_ROOT}/MPOcrVinCode/MPOcrVinCode.framework/MPOcrVinCode.bundle"
  install_resource "${PODS_ROOT}/MPScanSDK/MPScanSDK.framework/MPScanSDK.bundle"
  install_resource "${PODS_ROOT}/NBInside/NBInside.framework/NBInside.bundle"
  install_resource "${PODS_ROOT}/NebulaResource/NebulaResource.framework/NebulaResource.bundle"
  install_resource "${PODS_ROOT}/NebulaSDKPlugins/NebulaSDKPlugins.framework/NebulaSDKPlugins.bundle"
  install_resource "${PODS_ROOT}/TBDecodeSDK/TBDecodeSDK.framework/TBDecodeSDK.bundle"
  install_resource "${PODS_ROOT}/TBScanSDK/TBScanSDK.framework/Resources/TBScanSDK.bundle"
  install_resource "${PODS_ROOT}/TTTAttributedLabel/TTTAttributedLabel.framework/TTTAttributedLabel.bundle"
  install_resource "${PODS_ROOT}/Texel/Texel.framework/Texel.bundle"
  install_resource "${PODS_ROOT}/XMedia/XMedia.framework/XMedia.bundle"
  install_resource "${PODS_ROOT}/uep/uep.framework/uep.bundle"
fi
if [[ "$CONFIGURATION" == "Release" ]]; then
  install_resource "${PODS_ROOT}/APLongLinkService/APLongLinkService.framework/APLongLinkService.bundle"
  install_resource "${PODS_ROOT}/APMAftsCommon/APMAftsCommon.framework/APMAftsCommon.bundle"
  install_resource "${PODS_ROOT}/APMCDNImage/APMCDNImage.framework/APMCDNImage.bundle"
  install_resource "${PODS_ROOT}/APMFileTransfer/APMFileTransfer.framework/APMFileTransfer.bundle"
  install_resource "${PODS_ROOT}/APMHevc/APMHevc.framework/APMHevc.bundle"
  install_resource "${PODS_ROOT}/APMImageProcessing/APMImageProcessing.framework/APMImageProcessing.bundle"
  install_resource "${PODS_ROOT}/APMMediaCache/APMMediaCache.framework/APMMediaCache.bundle"
  install_resource "${PODS_ROOT}/APMUtils/APMUtils.framework/APMUtils.bundle"
  install_resource "${PODS_ROOT}/APMVideo/APMVideo.framework/APMVideo.bundle"
  install_resource "${PODS_ROOT}/APMVoiceManager/APMVoiceManager.framework/APMVoiceManager.bundle"
  install_resource "${PODS_ROOT}/APMap/APMap.framework/APMap.bundle"
  install_resource "${PODS_ROOT}/APMediaBasic/APMediaBasic.framework/APMediaBasic.bundle"
  install_resource "${PODS_ROOT}/APMobileFramework/APMobileFramework.framework/APMobileFramework.bundle"
  install_resource "${PODS_ROOT}/APMobileNetwork/APMobileNetwork.framework/APMobileNetwork.bundle"
  install_resource "${PODS_ROOT}/APMultimedia/APMultimedia.framework/APMultimedia.bundle"
  install_resource "${PODS_ROOT}/APRemoteLogging/APRemoteLogging.framework/APRemoteLogging.bundle"
  install_resource "${PODS_ROOT}/AUBase/AUBase.framework/AUBase.bundle"
  install_resource "${PODS_ROOT}/AUKeyboard/AUKeyboard.framework/AUKeyboard.bundle"
  install_resource "${PODS_ROOT}/AUTheme/AUTheme.framework/AUTheme.bundle"
  install_resource "${PODS_ROOT}/AliUpgradeCheckService/AliUpgradeCheckService.framework/AliUpgradeCheckService.bundle"
  install_resource "${PODS_ROOT}/AliUpgradeUI/AliUpgradeUI.framework/AliUpgradeUI.bundle"
  install_resource "${PODS_ROOT}/AntUI/AntUI.framework/APCommonUI.bundle"
  install_resource "${PODS_ROOT}/AntUIToken/AntUIToken.framework/AntUIToken.bundle"
  install_resource "${PODS_ROOT}/AriverComponent/AriverComponent.framework/AriverComponent.bundle"
  install_resource "${PODS_ROOT}/AriverKernel/AriverKernel.framework/AriverKernel.bundle"
  install_resource "${PODS_ROOT}/AriverMPNebulaAdapter/AriverMPNebulaAdapter.framework/MPInternalPlugins.bundle"
  install_resource "${PODS_ROOT}/AriverMPNebulaAdapter/AriverMPNebulaAdapter.framework/MPNebulaAdapter.bundle"
  install_resource "${PODS_ROOT}/AriverMap/AriverMap.framework/AriverMap.bundle"
  install_resource "${PODS_ROOT}/AriverNebulaAppBiz/AriverNebulaAppBiz.framework/NebulaAppBiz.bundle"
  install_resource "${PODS_ROOT}/AriverNebulaAppManager/AriverNebulaAppManager.framework/NebulaAppManager.bundle"
  install_resource "${PODS_ROOT}/AriverNebulaBiz/AriverNebulaBiz.framework/NebulaBiz.bundle"
  install_resource "${PODS_ROOT}/AriverNebulaBiz/AriverNebulaBiz.framework/NebulaPoseidon.bundle"
  install_resource "${PODS_ROOT}/AriverNebulaConfig/AriverNebulaConfig.framework/NebulaConfig.bundle"
  install_resource "${PODS_ROOT}/AriverRuntime/AriverRuntime.framework/AriverRuntime.bundle"
  install_resource "${PODS_ROOT}/AriverTinyAppCommon/AriverTinyAppCommon.framework/TinyAppCommon.bundle"
  install_resource "${PODS_ROOT}/AriverTinyappService/AriverTinyappService.framework/TinyappService.bundle"
  install_resource "${PODS_ROOT}/BEEAudioUtil/BEEAudioUtil.framework/BEEAudioUtil.bundle"
  install_resource "${PODS_ROOT}/BeeCapture/BeeCapture.framework/BeeCapture.bundle"
  install_resource "${PODS_ROOT}/BeeCityPicker/BeeCityPicker.framework/BeeCityPicker.bundle"
  install_resource "${PODS_ROOT}/BeeLocationPicker/BeeLocationPicker.framework/BeeLocationPicker.bundle"
  install_resource "${PODS_ROOT}/BeeMediaPlayer/BeeMediaPlayer.framework/BeeMediaPlayer.bundle"
  install_resource "${PODS_ROOT}/BeePhoto/BeePhoto.framework/BeePhoto.bundle"
  install_resource "${PODS_ROOT}/BeePhotoBrowser/BeePhotoBrowser.framework/BeePhotoBrowser.bundle"
  install_resource "${PODS_ROOT}/BeePhotoPicker/BeePhotoPicker.framework/BeePhotoPicker.bundle"
  install_resource "${PODS_ROOT}/BeeUtils/BeeUtils.framework/BeeUtils.bundle"
  install_resource "${PODS_ROOT}/MAMapKit/MAMapKit.framework/AMap.bundle"
  install_resource "${PODS_ROOT}/MPHotpatchSDK/MPHotpatchSDK.framework/DynamicDeployment.bundle"
  install_resource "${PODS_ROOT}/MPOcr/MPOcr.framework/MPOcr.bundle"
  install_resource "${PODS_ROOT}/MPOcrCarLicense/MPOcrCarLicense.framework/MPOcrCarLicense.bundle"
  install_resource "${PODS_ROOT}/MPOcrDriverLicense/MPOcrDriverLicense.framework/MPOcrDriverLicense.bundle"
  install_resource "${PODS_ROOT}/MPOcrDrivingLicense/MPOcrDrivingLicense.framework/MPOcrDrivingLicense.bundle"
  install_resource "${PODS_ROOT}/MPOcrGasMeter/MPOcrGasMeter.framework/MPOcrGasMeter.bundle"
  install_resource "${PODS_ROOT}/MPOcrPassPort/MPOcrPassPort.framework/MPOcrPassPort.bundle"
  install_resource "${PODS_ROOT}/MPOcrVinCode/MPOcrVinCode.framework/MPOcrVinCode.bundle"
  install_resource "${PODS_ROOT}/MPScanSDK/MPScanSDK.framework/MPScanSDK.bundle"
  install_resource "${PODS_ROOT}/NBInside/NBInside.framework/NBInside.bundle"
  install_resource "${PODS_ROOT}/NebulaResource/NebulaResource.framework/NebulaResource.bundle"
  install_resource "${PODS_ROOT}/NebulaSDKPlugins/NebulaSDKPlugins.framework/NebulaSDKPlugins.bundle"
  install_resource "${PODS_ROOT}/TBDecodeSDK/TBDecodeSDK.framework/TBDecodeSDK.bundle"
  install_resource "${PODS_ROOT}/TBScanSDK/TBScanSDK.framework/Resources/TBScanSDK.bundle"
  install_resource "${PODS_ROOT}/TTTAttributedLabel/TTTAttributedLabel.framework/TTTAttributedLabel.bundle"
  install_resource "${PODS_ROOT}/Texel/Texel.framework/Texel.bundle"
  install_resource "${PODS_ROOT}/XMedia/XMedia.framework/XMedia.bundle"
  install_resource "${PODS_ROOT}/uep/uep.framework/uep.bundle"
fi

mkdir -p "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}"
rsync -avr --copy-links --no-relative --exclude '*/.svn/*' --files-from="$RESOURCES_TO_COPY" / "${TARGET_BUILD_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}"
if [[ "${ACTION}" == "install" ]] && [[ "${SKIP_INSTALL}" == "NO" ]]; then
  mkdir -p "${INSTALL_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}"
  rsync -avr --copy-links --no-relative --exclude '*/.svn/*' --files-from="$RESOURCES_TO_COPY" / "${INSTALL_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}"
fi
rm -f "$RESOURCES_TO_COPY"

if [[ -n "${WRAPPER_EXTENSION}" ]] && [ "`xcrun --find actool`" ] && [ -n "${XCASSET_FILES:-}" ]
then
  # Find all other xcassets (this unfortunately includes those of path pods and other targets).
  OTHER_XCASSETS=$(find -L "$PWD" -iname "*.xcassets" -type d)
  while read line; do
    if [[ $line != "${PODS_ROOT}*" ]]; then
      XCASSET_FILES+=("$line")
    fi
  done <<<"$OTHER_XCASSETS"

  if [ -z ${ASSETCATALOG_COMPILER_APPICON_NAME+x} ]; then
    printf "%s\0" "${XCASSET_FILES[@]}" | xargs -0 xcrun actool --output-format human-readable-text --notices --warnings --platform "${PLATFORM_NAME}" --minimum-deployment-target "${!DEPLOYMENT_TARGET_SETTING_NAME}" ${TARGET_DEVICE_ARGS} --compress-pngs --compile "${BUILT_PRODUCTS_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}"
  else
    printf "%s\0" "${XCASSET_FILES[@]}" | xargs -0 xcrun actool --output-format human-readable-text --notices --warnings --platform "${PLATFORM_NAME}" --minimum-deployment-target "${!DEPLOYMENT_TARGET_SETTING_NAME}" ${TARGET_DEVICE_ARGS} --compress-pngs --compile "${BUILT_PRODUCTS_DIR}/${UNLOCALIZED_RESOURCES_FOLDER_PATH}" --app-icon "${ASSETCATALOG_COMPILER_APPICON_NAME}" --output-partial-info-plist "${TARGET_TEMP_DIR}/assetcatalog_generated_info_cocoapods.plist"
  fi
fi
